#
# XML Object parser.
#
# This package uses the xml library to parse
# the datastream and places all of the data into a
# series of nested associative arrays
#
# NOTE: Although the xml library supports multiple
#       simultaneous stream parsing, this package
#       funnels all parsing through a common set of routines.
#       Since the xml library uses callbacks that do not
#       indicate the stream that they pertain to, it is
#       imposible to process multiple streams with
#       the XMLObject package.

package provide XMLObject 2.2

package require xml
package require Tree 2.2

namespace eval ::XMLObject {

  namespace export setTrimChars parse
  namespace export create createChild name
  namespace export children child parent
  namespace export value list setValue 
  namespace export setTrimChars exist
  namespace export format


  set Debug 0
  set TrimSet "\n\t\r "

  set ParserName ""
  set WorkingNode {}
  set WorkingData {}
}

# ---------------------------------------------------------- format
# format  token
# format the xml object represented by token.
proc ::XMLObject::format { token } {
  variable output
  set output ""

  return [::Tree::format $token]
}

# ---------------------------------------------------------- setTrimChars
proc ::XMLObject::setTrimChars { trimChars } {
  variable TrimSet

  set TrimSet $trimChars
}

# ---------------------------------------------------------- create
# WARNING - backward compatability only
# Create an empty XML Object
# Return the token for this object
proc ::XMLObject::create { } {

  return [::Tree::create "root"]
}

# ---------------------------------------------------------- createChild
# WARNING - backward compatability only
# Create an empty XML child object from the node of token
# Return the token for this child object
# if an object of this name already exists, create a second node
# of the same name.
proc ::XMLObject::createChild { token name } {

  set newChild [::Tree::create $name]
  ::Tree::graft $token $newChild
  return $newChild
}

# ---------------------------------------------------------- children
# WARNING - backward compatability only
# return the list of children for the object token
proc ::XMLObject::children { token } {

  return [::Tree::children $token]
}

# ---------------------------------------------------------- child
# WARNING - backward compatability only
# return the list of children named name
proc ::XMLObject::child { token name } {

  return [::Tree::child $token $name]
}

# ---------------------------------------------------------- name
# WARNING - backward compatability only
# return the value of the node
proc ::XMLObject::name { token } {

  return [::Tree::name $token]
}

# ---------------------------------------------------------- value
# WARNING - backward compatability only
# return the value of the node
proc ::XMLObject::value { token } {

  return [::Tree::attribute $token value]
}

# ---------------------------------------------------------- list
# WARNING - backward compatability only
# return the list of values pertaining to the multiple
# children called name of token
proc ::XMLObject::list { token name} {

  return [::Tree::list $token $name]
}

# ---------------------------------------------------------- setValue
# WARNING - backward compatability only
# set the value of the node represented by token
proc ::XMLObject::setValue { token value } {

  return [::Tree::attribute $token value $value]
}

# ---------------------------------------------------------- parent
# WARNING - backward compatability only
# return the parent node for the object token
proc ::XMLObject::parent { token } {

  return [::Tree::parent $token]
}

# ---------------------------------------------------------- exist
# WARNING - backward compatability only
# exist   token  name ?childvar?
# return a 1 if the child exists. If childvar is included
# then set it to the token of the child var.
proc ::XMLObject::exist { token name args} {

  if {[llength $args] > 0} then {
    upvar [lindex $args 0] childNode
    return [::Tree::exist $token $name childNode]
  } else {
    return [::Tree::exist $token $name]
  }
}


# ---------------------------------------------------------- parse
# Create a parser for the data stream
proc ::XMLObject::parse { data } {
  variable WorkingNode
  variable WorkingData
  variable ParserObject
  variable ParserName
  
  if {$ParserName == ""} then {
    set ParserName [::xml::parser a]
  }
  $ParserName configure -elementstartcommand ::XMLObject::EStart \
                    -elementendcommand ::XMLObject::EEnd     \
                    -characterdatacommand ::XMLObject::CData \
                    -final 1                             \
                    -reportempty 1                       \
                    -processinginstructioncommand ::XMLObject::PInst  \
                    -xmldeclcommand ::XMLObject::XMLDecl              \
                    -doctypecommand ::XMLObject::HandleDocType        \
                    -commentcommand ::XMLObject::HandleComment


  set XMLTree [::Tree::create root]
  set WorkingNode $XMLTree
  set WorkingData {}

  $ParserName reset
  if {[catch {$ParserName parse $data} msg]} then {
    puts $msg
  }
  return $XMLTree
}

# ---------------------------------------------------------- EStart
proc ::XMLObject::EStart { name attlist } {
  variable WorkingNode
  variable WorkingData
  variable Debug
  variable TrimSet

#  puts "Start - $name ($WorkingNode)"
  
  ::Tree::appendAttribute $WorkingNode value [string trim $WorkingData $TrimSet]
  set WorkingData ""

  set WorkingNode [::Tree::graft $WorkingNode [::Tree::create $name]]

  if {$Debug == 1} then {
    puts "START - $name  ($WorkingNode)"
    foreach {name value} $attlist {
      puts stderr "        $name = $value"
    }
  }
}

# ---------------------------------------------------------- EEnd
proc ::XMLObject::EEnd { name } {
  variable WorkingNode
  variable WorkingData
  variable Debug
  variable TrimSet

#  puts "End - $name ($WorkingNode)"

  ::Tree::appendAttribute $WorkingNode value [string trim $WorkingData $TrimSet]

  set WorkingNode [::Tree::parent $WorkingNode]
  set WorkingData [::Tree::attribute $WorkingNode value]
  
  if {$Debug == 1} then {
    puts stderr "END   - [string tolower $name] ($WorkingNode)"
    puts "\[::Tree::value $WorkingNode\]"
  }
}

# ---------------------------------------------------------- CData 
proc ::XMLObject::CData { data } {
  variable WorkingData
  variable Debug

  if {$Debug == 1} then {
    puts stderr "DATA  - ([string length $data]) $data"
  }
  append WorkingData $data
}

# ---------------------------------------------------------- PInst 
proc ::XMLObject::PInst { target data } {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "PInst  - $target : $data"
  }
}

# ---------------------------------------------------------- XMLDecl 
proc ::XMLObject::XMLDecl { version encoding standalone } {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "XML Declaration ==> version $version encoding $encoding standalone $standalone"
  }
}

# ---------------------------------------------------------- HandleDocType 
proc ::XMLObject::HandleDocType {docelement publicID systemID internalDTD} {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "Doc Type Declaration ==> doc element $docelement, internal DTD subset $internalDTD"
  }
}

# ---------------------------------------------------------- HandleComment 
proc ::XMLObject::HandleComment {data} {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "Comment ==> $data"
  }
}
