#
# XML Object parser.
#
# This package uses the xml library to parse
# the datastream and places all of the data into a
# series of nested associative arrays
#
# NOTE: Although the xml library supports multiple
#       simultaneous stream parsing, this package
#       funnels all parsing through a common set of routines.
#       Since the xml library uses callbacks that do not
#       indicate the stream that they pertain to, it is
#       imposible to process multiple streams with
#       the XMLObject package.

package provide XMLObject 3.4

package require xml
package require Tree 2.2

namespace eval ::XMLObject {

  namespace export setTrimChars parse
  namespace export format


  set Debug 0
  set TrimSet "\n\t\r "

  set ParserName ""
  set WorkingNode {}
  set WorkingData {}
}

# ---------------------------------------------------------- format
# format  token
# format the xml object represented by token.
proc ::XMLObject::format { token } {
  variable output
  set output ""

  return [::Tree::format $token]
}

# ---------------------------------------------------------- setTrimChars
proc ::XMLObject::setTrimChars { trimChars } {
  variable TrimSet

  set TrimSet $trimChars
}

# ---------------------------------------------------------- parse
# Create a parser for the data stream
proc ::XMLObject::parse { data } {
  variable WorkingNode
  variable WorkingData
  variable ParserObject
  variable ParserName
  
  if {$ParserName == ""} then {
    set ParserName [::xml::parser a]
  }
  $ParserName configure -elementstartcommand ::XMLObject::EStart \
                    -elementendcommand ::XMLObject::EEnd     \
                    -characterdatacommand ::XMLObject::CData \
                    -final 1                             \
                    -reportempty 1                       \
                    -processinginstructioncommand ::XMLObject::PInst  \
                    -xmldeclcommand ::XMLObject::XMLDecl              \
                    -doctypecommand ::XMLObject::HandleDocType        \
                    -commentcommand ::XMLObject::HandleComment


  set XMLTree [::Tree::create root]
  set WorkingNode $XMLTree
  set WorkingData {}

  if {[catch {$ParserName parse $data} msg]} then {
    puts $msg
  }
  catch {$ParserName free $ParserName}
  return $XMLTree
}

# ---------------------------------------------------------- EStart
proc ::XMLObject::EStart { name attlist } {
  variable WorkingNode
  variable WorkingData
  variable Debug
  variable TrimSet
  
  ::Tree::appendAttribute $WorkingNode value [string trim $WorkingData $TrimSet]
  set WorkingData ""

  set WorkingNode [::Tree::graft $WorkingNode [::Tree::create $name]]
  foreach {name value} $attlist {
    ::Tree::attribute $WorkingNode $name $value
  }

  if {$Debug == 1} then {
    puts "START - $name  ($WorkingNode)"
    foreach {name value} $attlist {
      puts stderr "        $name = $value"
    }
  }
}

# ---------------------------------------------------------- EEnd
proc ::XMLObject::EEnd { name } {
  variable WorkingNode
  variable WorkingData
  variable Debug
  variable TrimSet

#  puts "End - $name ($WorkingNode)"

  ::Tree::appendAttribute $WorkingNode value [string trim $WorkingData $TrimSet]

  set WorkingNode [::Tree::parent $WorkingNode]
  set WorkingData [::Tree::attribute $WorkingNode value]
  
  if {$Debug == 1} then {
    puts stderr "END   - [string tolower $name] ($WorkingNode)"
    puts "\[::Tree::value $WorkingNode\]"
  }
}

# ---------------------------------------------------------- CData 
proc ::XMLObject::CData { data } {
  variable WorkingData
  variable Debug

  if {$Debug == 1} then {
    puts stderr "DATA  - ([string length $data]) $data"
  }
  append WorkingData $data
}

# ---------------------------------------------------------- PInst 
proc ::XMLObject::PInst { target data } {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "PInst  - $target : $data"
  }
}

# ---------------------------------------------------------- XMLDecl 
proc ::XMLObject::XMLDecl { version encoding standalone } {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "XML Declaration ==> version $version encoding $encoding standalone $standalone"
  }
}

# ---------------------------------------------------------- HandleDocType 
proc ::XMLObject::HandleDocType {docelement publicID systemID internalDTD} {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "Doc Type Declaration ==> doc element $docelement, internal DTD subset $internalDTD"
  }
}

# ---------------------------------------------------------- HandleComment 
proc ::XMLObject::HandleComment {data} {
  variable Debug

  if {$Debug == 1} then {
    puts stderr "Comment ==> $data"
  }
}
